<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\ProjectModel;

class ProjectsController extends BaseController
{
    protected $projectModel;

    public function __construct()
    {
        $this->projectModel = new ProjectModel();
    }

    public function index()
    {
        $search = $this->request->getGet('search') ?? '';
        $filter = [
            'jenis_project' => $this->request->getGet('jenis_project') ?? '',
            'tahun' => $this->request->getGet('tahun') ?? ''
        ];
        $sort = $this->request->getGet('sort') ?? 'urutan';
        $order = $this->request->getGet('order') ?? 'ASC';
        $page = (int)($this->request->getGet('page') ?? 1);

        $builder = $this->projectModel->searchAndFilter($search, $filter, $sort, $order);
        $total = $builder->countAllResults(false);
        
        $projects = $builder->get(10, ($page - 1) * 10)->getResultArray();

        $pager = \Config\Services::pager();
        $pager->store('projects', $page, 10, $total);

        $data = [
            'title' => 'Kelola Project',
            'projects' => $projects,
            'pager' => $pager,
            'search' => $search,
            'filter' => $filter,
            'sort' => $sort,
            'order' => $order
        ];

        return view('admin/projects/index', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Project'
        ];
        return view('admin/projects/create', $data);
    }

    public function store()
    {
        $rules = [
            'nama_project' => 'required|min_length[3]|max_length[200]',
            'jenis_project' => 'required|in_list[Web,Mobile,Game,Desktop,Lainnya]',
            'foto' => 'permit_empty|uploaded[foto]|max_size[foto,2048]|ext_in[foto,jpg,jpeg,png,gif]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $fileFoto = $this->request->getFile('foto');
        $fotoName = null;

        if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
            $fotoName = $fileFoto->getRandomName();
            $fileFoto->move(WRITEPATH . 'uploads/projects/', $fotoName);
        }

        $data = [
            'nama_project' => $this->request->getPost('nama_project'),
            'jenis_project' => $this->request->getPost('jenis_project'),
            'teknologi' => $this->request->getPost('teknologi'),
            'tahun' => $this->request->getPost('tahun'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'link_project' => $this->request->getPost('link_project'),
            'foto' => $fotoName,
            'urutan' => (int)($this->request->getPost('urutan') ?? 0)
        ];

        if ($this->projectModel->insert($data)) {
            return redirect()->to('admin/projects')->with('success', 'Project berhasil ditambahkan!');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal menambahkan project!');
        }
    }

    public function edit($id)
    {
        $project = $this->projectModel->find($id);
        
        if (!$project) {
            return redirect()->to('admin/projects')->with('error', 'Project tidak ditemukan!');
        }

        $data = [
            'title' => 'Edit Project',
            'project' => $project
        ];
        return view('admin/projects/edit', $data);
    }

    public function update($id)
    {
        $project = $this->projectModel->find($id);
        
        if (!$project) {
            return redirect()->to('admin/projects')->with('error', 'Project tidak ditemukan!');
        }

        $rules = [
            'nama_project' => 'required|min_length[3]|max_length[200]',
            'jenis_project' => 'required|in_list[Web,Mobile,Game,Desktop,Lainnya]',
            'foto' => 'permit_empty|uploaded[foto]|max_size[foto,2048]|ext_in[foto,jpg,jpeg,png,gif]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $fileFoto = $this->request->getFile('foto');
        
        $data = [
            'nama_project' => $this->request->getPost('nama_project'),
            'jenis_project' => $this->request->getPost('jenis_project'),
            'teknologi' => $this->request->getPost('teknologi'),
            'tahun' => $this->request->getPost('tahun'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'link_project' => $this->request->getPost('link_project'),
            'urutan' => (int)($this->request->getPost('urutan') ?? 0)
        ];

        if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
            if ($project['foto'] && file_exists(WRITEPATH . 'uploads/projects/' . $project['foto'])) {
                unlink(WRITEPATH . 'uploads/projects/' . $project['foto']);
            }
            $fotoName = $fileFoto->getRandomName();
            $fileFoto->move(WRITEPATH . 'uploads/projects/', $fotoName);
            $data['foto'] = $fotoName;
        }

        if ($this->projectModel->update($id, $data)) {
            return redirect()->to('admin/projects')->with('success', 'Project berhasil diupdate!');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal mengupdate project!');
        }
    }

    public function delete($id)
    {
        $project = $this->projectModel->find($id);
        
        if (!$project) {
            return redirect()->to('admin/projects')->with('error', 'Project tidak ditemukan!');
        }

        if ($project['foto'] && file_exists(WRITEPATH . 'uploads/projects/' . $project['foto'])) {
            unlink(WRITEPATH . 'uploads/projects/' . $project['foto']);
        }

        if ($this->projectModel->delete($id)) {
            return redirect()->to('admin/projects')->with('success', 'Project berhasil dihapus!');
        } else {
            return redirect()->to('admin/projects')->with('error', 'Gagal menghapus project!');
        }
    }
}


